<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Convalidacion;
use App\Models\Postulante;
use App\Models\Matricula;
use App\Models\MatriculaCurso;
use App\Models\Periodo;
use App\Models\Curso;
use App\Models\NotaFinal;
use PDF;
use App\Models\Instituto;
class ConvalidacionController extends Controller
{
    public function index()
    {
        return view('convalidaciones.index');
    }

    public function buscarPorDni(Request $request)
    {
        $dni = $request->dni;

        $postulante = Postulante::where('dni', $dni)->first();

        if (!$postulante) {
            return response()->json(['error' => 'Estudiante no encontrado.'], 404);
        }

        $periodoActivo = Periodo::where('activo', '1')->first();

        $matricula = Matricula::with(['planEstudio', 'semestre'])
            ->where('postulante_id', $postulante->id_postulante)
            ->where('periodo_id', $periodoActivo->id ?? 0)
            ->first();

        if (!$matricula) {
            return response()->json(['error' => 'No se encontró matrícula para el periodo activo.'], 404);
        }

        $cursos = MatriculaCurso::with('curso')
            ->where('matricula_id', $matricula->id)
            ->get();

        return response()->json([
            'postulante' => $postulante,
            'matricula' => $matricula,
            'cursos' => $cursos,
            'periodo' => $periodoActivo,
        ]);
    }

    public function store(Request $request)
{
    // 1. Validar datos
    $request->validate([
        'estudiante_id' => 'required|exists:postulantes,id_postulante',
        'curso_destino_id' => 'required|exists:cursos,id',
        'curso_origen' => 'required|string',
        'institucion_origen' => 'required|string',
        'nota_origen' => 'required|numeric|min:0|max:20',
        'periodo_id' => 'required|exists:periodos,id',
        'fecha_convalidacion' => 'required|date',
        'observaciones' => 'nullable|string',
    ]);

    // 2. Registrar convalidación
    $convalidacion = Convalidacion::create($request->all());

    // 3. Buscar matrícula
    $matricula = Matricula::where('postulante_id', $request->estudiante_id)
        ->where('periodo_id', $request->periodo_id)
        ->first();

    if (!$matricula) {
        return response()->json(['error' => 'Matrícula no encontrada para el periodo.'], 404);
    }

    // 4. Marcar curso como convalidado en la matrícula si existe
    $matriculaCurso = MatriculaCurso::where('matricula_id', $matricula->id)
        ->where('curso_id', $request->curso_destino_id)
        ->first();

    if ($matriculaCurso) {
        $matriculaCurso->estado = 'Convalidado';
        $matriculaCurso->save();
    }

    // 5. Registrar o actualizar la nota final
    NotaFinal::updateOrCreate(
        [
            'estudiante_id' => $request->estudiante_id,
            'curso_id' => $request->curso_destino_id
        ],
        [
            'promedio' => $request->nota_origen,
            'usuario' => auth()->user()->usuario ?? 'sistema'
        ]
    );

    return response()->json(['message' => 'Convalidación registrada correctamente.']);
}
public function listado()
{
    $convalidaciones = Convalidacion::with(['estudiante', 'cursoDestino', 'periodo'])
        ->latest()
        ->get();

    return view('convalidaciones.listado', compact('convalidaciones'));
}
public function reporte(Request $request)
{
    $request->validate([
        'fecha_inicio' => 'required|date',
        'fecha_fin' => 'required|date|after_or_equal:fecha_inicio',
    ]);

    $convalidaciones = Convalidacion::with(['estudiante', 'cursoDestino', 'periodo'])
        ->whereBetween('fecha_convalidacion', [$request->fecha_inicio, $request->fecha_fin])
        ->get();

    $instituto = Instituto::first();

    // Pasar fechas explícitamente a la vista
    $fechaInicio = $request->fecha_inicio;
    $fechaFin = $request->fecha_fin;

    $pdf = PDF::loadView('convalidaciones.reporte_pdf', compact('convalidaciones', 'instituto', 'fechaInicio', 'fechaFin'))
        ->setPaper('a4', 'landscape');

    return $pdf->stream('reporte_convalidaciones.pdf');
}


}
